package ch.frankel.blog.hibernate;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertNotNull;
import static org.testng.Assert.assertNotSame;
import static org.testng.Assert.assertNull;
import static org.testng.Assert.assertSame;
import static org.testng.Assert.assertTrue;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;

import junitx.util.PrivateAccessor;

import org.hibernate.Session;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import ch.frankel.blog.hibernate.model.Owner;

/**
 * Test class to check differences between <code>load()</code> and
 * <code>get()</code>.
 * 
 * @author Nicolas Frnkel
 */
public class TestPart4 extends AbstractTest {

	private PrintStream original;

	private ByteArrayOutputStream testStream;

	/**
	 * Redirects the sysout to a referenced stream for testing purposes.
	 */
	@BeforeMethod
	protected void setUp() {

		testStream = new ByteArrayOutputStream();

		original = System.out;

		System.setOut(new PrintStream(testStream));

		factory.getCurrentSession().beginTransaction();
	}

	/**
	 * Resets the sysout and the current session.
	 */
	@AfterMethod
	protected void tearDown() {

		factory.getCurrentSession().clear();

		factory.getCurrentSession().getTransaction().rollback();

		System.setOut(original);
	}

	/**
	 * Test the <code>get()</code> method.
	 * 
	 * @throws NoSuchFieldException
	 */
	@Test
	public void testGetClass() throws NoSuchFieldException {

		Session session = factory.getCurrentSession();

		Owner owner = (Owner) session.get(Owner.class, 1);

		// Test the class of the object
		assertSame(owner.getClass(), Owner.class);

		// Test the select order
		assertTrue(new String(testStream.toByteArray()).contains("select"));

		// Test if the fields are really there
		assertNotNull(PrivateAccessor.getField(owner, "date"));
		assertNotNull(PrivateAccessor.getField(owner, "id"));
	}

	/**
	 * Test the <code>load()</code> method.
	 * 
	 * @throws NoSuchFieldException
	 */
	@Test
	public void testLoad() throws NoSuchFieldException {

		Session session = factory.getCurrentSession();

		Owner owner = (Owner) session.load(Owner.class, 1);

		// Test the class of the object
		assertNotSame(owner.getClass(), Owner.class);

		// Test the select order
		assertEquals(new String(testStream.toByteArray()), "");

		// Test if the fields are really there
		assertNull(PrivateAccessor.getField(owner, "date"));
		assertNull(PrivateAccessor.getField(owner, "id"));
	}

	/**
	 * Test the caching between <code>load()</code> and <code>get()</code>.
	 * 
	 * @throws NoSuchFieldException
	 */
	@Test
	public void testLoadBeforeGetCaching() throws NoSuchFieldException {

		Session session = factory.getCurrentSession();

		// Loads the reference
		session.load(Owner.class, 1);

		// Gets the real entity, but looks for it in the cache and gets
		// the reference instead
		Owner owner = (Owner) session.get(Owner.class, 1);

		// Test the class of the object
		assertNotSame(owner.getClass(), Owner.class);

		// Test if the fields are really there
		assertNull(PrivateAccessor.getField(owner, "date"));
		assertNull(PrivateAccessor.getField(owner, "id"));
	}

	/**
	 * Test the caching between <code>load()</code> and <code>get()</code>.
	 * 
	 * @throws NoSuchFieldException
	 */
	@Test
	public void testGetBeforeLoadCaching() throws NoSuchFieldException {

		Session session = factory.getCurrentSession();

		// Gets the object
		session.get(Owner.class, 1);

		// Loads the reference, but looks for it in the cache and loads
		// the real entity instead
		Owner owner = (Owner) session.load(Owner.class, 1);

		// Test the class of the object
		assertSame(owner.getClass(), Owner.class);

		// Test if the fields are really there
		assertNotNull(PrivateAccessor.getField(owner, "date"));
		assertNotNull(PrivateAccessor.getField(owner, "id"));
	}
}
